unit common;

(* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version 
 * 1.1 (the "License"); you may not use this file except in compliance with 
 * the License. You may obtain a copy of the License at 
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is a utility unit for Lightning.
 *
 * The Initial Developer of the Original Code is
 * Tomsoft.
 * Portions created by Tomsoft are Copyright (C) 2009
 * Tomsoft. All Rights Reserved.
 *
 * Contributor(s):
 *   Tommy Sools <tom@sools.com>
 *
 * ***** END LICENSE BLOCK ***** *)

//{$define lite}

interface

uses
  Windows, SysUtils;

const
  levelMin        = 0;
  levelMax        = 5119;
  dmxLevelMin     = 0;
  dmxLevelMax     = 255;
  fftMin          = 1;
  fftMax          = 1024;
  enabledBoundary = 0;
  maxChannels     = 64;
  detailsTop      = 4;
  detailsLeft     = 4;
  detailsWidth    = 728;
  detailsHeight   = 176;
  maxLength       = 65535;

type
  TLevel      = levelMin..levelMax;
  TDMXLevel = dmxLevelMin..dmxLevelMax;
  TSpectrum   = array[fftMin..fftMax] of TLevel;
  TOpenFile   = procedure(fileName: PChar); cdecl;
  TLightningModule = record
    handle:   THandle;
    title:    PChar;
    settings: Boolean;
    width, height, inputTop, infoTop, outputTop, inputs, outputs: Integer;
    inputTooltip, outputTooltip: PChar;
    details:  Pointer;
    openFile: TOpenFile;
    reserved: Pointer;
  end;
  PLightningModule = ^TLightningModule;
  TNextScene = procedure; cdecl;
  TNextProgram = procedure; cdecl;
  TSetFadeTime = procedure(fadeTime: Integer); cdecl;
  TLightningModule2 = record
    scenic:      Boolean; // not used anymore
    nextScene:   TNextScene;
    nextProgram: TNextProgram;
    setFadeTime: TSetFadeTime;
    reserved:    Pointer;
  end;
  PLightningModule2 = ^TLightningModule2;
  TNextProgramEx = procedure(index: Integer; abs: Boolean); cdecl;
  TLightningModule3 = record
    nextProgramEx: TNextProgramEx;
    reserved:      Pointer;
  end;
  PLightningModule3 = ^TLightningModule3;
  TModuleInit      = function(name: PChar; nameLength: DWORD): DWORD; cdecl;
  TModuleFinish    = procedure(); cdecl;
  TModuleCreate    = function(controlHandle: THandle): PLightningModule; cdecl;
  TModuleDestroy   = procedure(lightningModule: PLightningModule); cdecl;
  TModuleAction    = function(lightningModule: PLightningModule; input, output: Integer; inputConnected, outputConnected: PChar; var bridge: TSpectrum): Boolean; cdecl;
  TModuleShow      = procedure(lightningModule: PLightningModule; handle: THandle); cdecl;
  TModuleHide      = procedure(lightningModule: PLightningModule); cdecl;
  TModuleOpen      = procedure(lightningModule: PLightningModule; settings: PChar); cdecl;
  TModuleSave      = function(lightningModule: PLightningModule; settings: PChar; settingsLength: DWORD): DWORD; cdecl;
  TModuleOpenScene = procedure(lightningModule: PLightningModule; settings: PChar; fadeTime: Cardinal; reserved: Pointer); cdecl;
  TModuleSaveScene = function(lightningModule: PLightningModule; settings: PChar; settingsLength: DWORD): DWORD; cdecl;
  TModuleKeyboard  = procedure(lightningModule: PLightningModule; nVirtKey: LongInt; lKeyData: Cardinal); cdecl;

var
  tickFreq: Int64 = 0;

procedure createSpectrum(var bridge: TSpectrum; level: TLevel);
function  spectrumAverage(var bridge: TSpectrum; startFrequency, stopFrequency: Integer): TLevel;
procedure parse(var text: string; delimiter: string; var chunk: string); overload;
function  parse(var text: string; delimiter: string): string; overload;
function  return(text: string; dest: PChar; destLength: DWORD): DWORD;
function  booleanToString(b: Boolean): string;
procedure tickInit;
function  tickCount: Int64;
function  getConnectorCount(connectionString: string): Integer;
function  getLastConnector(connectionString: string): Integer;

implementation

procedure createSpectrum(var bridge: TSpectrum; level: TLevel);
var
  i: Integer;
begin
  {$ifndef lite}
  for i := 1 to fftMax do
    bridge[i] := level;
  {$endif}
  {$ifdef lite}
  bridge[1] := level;
  {$endif}
end;

function  spectrumAverage(var bridge: TSpectrum; startFrequency, stopFrequency: Integer): TLevel;
{$ifndef lite}
var
  i, j: Integer;
{$endif}
begin
{$ifndef lite}
  j := 0;
  for i := startFrequency to stopFrequency do
    Inc(j, bridge[i]);
  Result := Round(j / (stopFrequency - startFrequency + 1));
{$endif}
{$ifdef lite}
  Result := bridge[1];
{$endif}
end;

procedure parse(var text: string; delimiter: string; var chunk: string);
var
  position: Integer;
begin
  position := Pos(delimiter, text);
  if position = 0 then
  begin
    chunk := text;
    text  := '';
  end
  else
  begin
    chunk := Copy(text, 1, position - 1);
    text  := Copy(text, position + 1, Length(text));
  end;
end;

function  parse(var text: string; delimiter: string): string;
var
  position: Integer;
begin
  position := Pos(delimiter, text);
  if position = 0 then
  begin
    Result := text;
    text   := '';
  end
  else
  begin
    Result := Copy(text, 1, position - 1);
    text   := Copy(text, position + 1, Length(text));
  end;
end;

function  return(text: string; dest: PChar; destLength: DWORD): DWORD;
begin
  Result := StrLen(StrPLCopy(dest, text, destLength));
end;

function  booleanToString(b: Boolean): string;
begin
  if b then
    Result := '1'
  else
    Result := '';
end;

procedure tickInit;
begin
  QueryPerformanceFrequency(tickFreq);
end;

function  tickCount: Int64;
var
  ticks: Int64;
begin
  if tickFreq > 0 then
  begin
    if QueryPerformanceCounter(ticks) then
      Result := Round(ticks / (tickFreq / 1000))
    else
    begin
      tickFreq := 0;
      Result   := GetTickCount;
    end;
  end
  else
    Result := GetTickCount;
end;

function  getConnectorCount(connectionString: string): Integer;
begin
  Result := 0;
  while connectionString <> '' do
  begin
    Delete(connectionString, 1, Pos('|', connectionString));
    Inc(Result);
  end;
end;

function  getLastConnector(connectionString: string): Integer;
var
  counter: Integer;
begin
  counter := 0;
  Result := 0;
  while connectionString <> '' do
  begin
    if parse(connectionString, '|') <> '' then
      Result := counter;
    Inc(counter);
  end;
end;

end.
