; ***** BEGIN LICENSE BLOCK *****
; Version: MPL 1.1
;
; The contents of this file are subject to the Mozilla Public License Version 
; 1.1 (the "License"); you may not use this file except in compliance with 
; the License. You may obtain a copy of the License at 
; http://www.mozilla.org/MPL/
;
; Software distributed under the License is distributed on an "AS IS" basis,
; WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
; for the specific language governing rights and limitations under the
; License.
;
; The Original Code is firmware for the Demux12 device.
;
; The Initial Developer of the Original Code is
; Tomsoft.
; Portions created by Tomsoft are Copyright (C) 2009
; Tomsoft. All Rights Reserved.
;
; Contributor(s):
;   Tommy Sools <tom@sools.com>
;
; ***** END LICENSE BLOCK *****

; V1.3:
; improved switched output driver code
; use pin RB5 instead of RD7 for switch matrix send
; added support for power/signal led on pin RD7
; added watchdog timer

; V1.2:
; overal code cleanup
; doubled switch matrix delay
; temporarily disabled "DMX address" switch matrix; the DMX address is fixed to 28

; V1.1:
; fixed addressing issue, causing 1 switch channel to be impossible to use

; V1.0:
; initial version

; CPU:		PIC18F4455
; Speed:	20 MHz quartz
; 1 cycle = 4 clock pulses = 0.2 microsec

; RB0 = "DMX address" switch matrix send a1 a2 a3
; RB1 = "DMX address" switch matrix send a4 a5 a6
; RB2 = "DMX address" switch matrix send a7 a8 a9
; RD0 = "DMX address" switch matrix return a1 a4 a7
; RD1 = "DMX address" switch matrix return a2 a5 a8
; RD2 = "DMX address" switch matrix return a3 a6 a9

; RB3 = "output override" switch matrix send b1 b2 b3 b4
; RB4 = "output override" switch matrix send b5 b6 b7 b8
; RB5 = "output override" switch matrix send b9 b10 b11 b12
; RD3 = "output override" switch matrix return b1 b5 b9
; RD4 = "output override" switch matrix return b2 b6 b10
; RD5 = "output override" switch matrix return b3 b7 b11
; RD6 = "output override" switch matrix return b4 b8 b12

; RC2 = switched output 1: power outlet 1
; RC1 = switched output 2: power outlet 2
; RC0 = switched output 3: power outlet 3
; RE2 = switched output 4: power outlet 4
; RE1 = switched output 5: multi A-1, power outlet 1
; RE0 = switched output 6: multi A-2, power outlet 2
; RA5 = switched output 7: multi A-3, power outlet 3
; RA4 = switched output 8: multi A-4, side power outlet
; RA3 = switched output 9: multi B-1, power outlet 1
; RA2 = switched output 10: multi B-2, power outlet 2
; RA1 = switched output 11: multi B-3, power outlet 3
; RA0 = switched output 12: multi B-4, side power outlet

; RD7 = power LED/DMX activity LED

; timer 0: detection of DMX signal loss timeout
; timer 1: switch matrix cycle (future)

#include p18f4455.inc

		radix decimal

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

		__config	_CONFIG1L,	0x00	; 0x300000 ; clock source = OSC1/OSC2, postscaler = /1
		__config	_CONFIG1H,	0x0D	; 0x300001 ; oscillator = HS: USB-HS
		__config	_CONFIG2L,	0x07	; 0x300002 ; USB voltage regulator disabled, brown out voltage 4.5 V
		__config	_CONFIG2H,	_WDT_ON_2H & _WDTPS_32768_2H	; 0x300003 ; watchdog timer enabled, watchdog prescaler 1:32768

		__config	_CONFIG3H,	0x81	; 0x300005
		__config	_CONFIG4L,	0x81	; 0x300006 ; disable low voltage programming, disable hardware debugging

		__config	_CONFIG5L,	0x0F	; 0x300008
		__config	_CONFIG5H,	0xC0	; 0x300009
		__config	_CONFIG6L,	0x0F	; 0x30000A
		__config	_CONFIG6H,	0xE0	; 0x30000B
		__config	_CONFIG7L,	0x0F	; 0x30000C
		__config	_CONFIG7H,	0x40	; 0x30000D

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; variables for delay routine
var1							equ		0x00
var2							equ		0x01
var3							equ		0x02

;  memories for "DMX address" switch matrix
addressh						equ		0x03
addressl						equ		0x04
tempaddressl					equ		0x05

; memories for "output override" switch matrix
outputoverride1to4				equ		0x06
outputoverride5to12				equ		0x07

; memories for switched outputs
switches1to4					equ		0x08
  ; bit 3 = output switch 1
  ; bit 2 = output switch 2
  ; bit 1 = output switch 3
  ; bit 0 = output switch 4
switches5to12					equ		0x09
  ; bit 7 = output switch 5
  ; bit 6 = output switch 6
  ; bit 5 = output switch 7
  ; bit 4 = output switch 8
  ; bit 3 = output switch 9
  ; bit 2 = output switch 10
  ; bit 1 = output switch 11
  ; bit 0 = output switch 12
switches1to4temp				equ		0x0A
switches5to12temp				equ		0x0B

; reserved for use by interrupt handler
temp_wreg						equ		0x0C
temp_status						equ		0x0D
temp_bsr						equ		0x0E
dmx_state						equ		0x0F
countdownaddressh				equ		0x10
countdownaddressl				equ		0x11

; DMX values table
dmx_values_length				equ		12
dmx_values_first				equ		0x20
dmx_values_last					equ		dmx_values_first + dmx_values_length - 1

; countdown before DMX signal is assumed lost
dmx_signallostcountdown_init	equ		3
dmx_signallostcountdown			equ		0x12

; states for DMX state machine
dmx_state_waitforsignal			equ		0
dmx_state_waitforbreak			equ 	2
dmx_state_waitforstartcode		equ		4
dmx_state_waitforaddress		equ 	6
dmx_state_waitfordata			equ 	8

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

dmx_setstate:	macro	state
; pre: (the interrupt handler is executing) and ((state % 2) = 0) and (state <= 8);
; post: the DMX state machine's state equals state;
		MOVLW	state
		MOVWF	dmx_state,ACCESS
		endm

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

dmx_applyvalue:	macro	dmx_channel, destinationfile, destinationbit
; pre:	(WREG = 127) AND (0 <= dmx_channel < 12) AND (0 <= destinationbit <= 7)
; post:	if the table entry at index dmx_channel in the dmx_values table was > 127 then
;		(the specified bit destinationbit in file destinationfile was enabled)
;		else (the specified bit destinationbit in file destinationfile was disabled);
		CPFSGT	dmx_values_first + dmx_channel,ACCESS
		BRA		$ + 6
		; enable switch
		BSF		destinationfile,destinationbit,ACCESS
		BRA		$ + 4
		; disable switch
		BCF		destinationfile,destinationbit,ACCESS
		endm

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

		org		0x00000000
		; NOP required here for hardware debugging.
		NOP
		BRA		init

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

		org		0x00000008			; compatibility mode interrupts

		; store status registers for safekeeping
		MOVFF	STATUS,temp_status
		MOVFF	BSR,temp_bsr
		MOVWF	temp_wreg,ACCESS

		; check if this is a timer 0 interrupt
		BTFSS	INTCON,TMR0IF,ACCESS
		; this isn't a timer 0 interrupt, so it is a serial interrupt
		BRA		int_serial
		; this is a timer 0 interrupt
int_timer0:
		; clear timer 0 interrupt flag
		BCF		INTCON,TMR0IF,ACCESS
		; check if the signal lost countdown has finished
		TSTFSZ	dmx_signallostcountdown,ACCESS
		BRA		int_signalnotlost
int_signallost:
		; DMX signal lost, enable power led.
		BCF		LATD,7,ACCESS
		dmx_setstate dmx_state_waitforsignal
		BRA		int_end
int_signalnotlost:
		DECF	dmx_signallostcountdown,F,ACCESS
		BRA		int_end

int_serial:
		; reset signallostcountdown.
		MOVLW	dmx_signallostcountdown_init
		MOVWF	dmx_signallostcountdown,ACCESS
		; check if this could be a break.
		BTFSS	RCSTA,FERR,ACCESS
		BRA		int_nobreak
int_break:
		; verify that the received data is really zero.
		BTFSC	RCSTA,RX9D,ACCESS
		BRA		int_ignorebyte					; error: 9th data bit is not zero.
		TSTFSZ	RCREG, ACCESS
		BRA		int_end							; error: received data byte is not zero.
		dmx_setstate dmx_state_waitforstartcode	; success: received data is zero.
		; DMX signal present, blink power led.
		BTG		LATD,7,ACCESS
		BRA		int_end
int_nobreak:
		; switch (dmx_state) {
		MOVF	dmx_state,W,ACCESS
		ADDWF	PCL,F,ACCESS
		BRA		int_waitforsignal
		BRA		int_waitforbreak
		BRA		int_waitforstartcode
		BRA		int_waitforaddress
		BRA		int_waitfordata
		; }

int_waitforsignal:
		dmx_setstate dmx_state_waitforbreak
int_waitforbreak:
int_ignorebyte:
		MOVF	RCREG,W,ACCESS
		BRA		int_end

int_waitforstartcode:
		; check that the startcode is zero
		TSTFSZ	RCREG,ACCESS
		BRA		int_invalidstartcode
int_validstartcode:
		dmx_setstate dmx_state_waitforaddress
		; check if address is zero (this is a special case in which no address countdown is neccesary)
		; address 0 is the same as address 1
		TSTFSZ	addressh,ACCESS
		; address is not zero
		BRA		int_addressnotzero
		TSTFSZ	addressl,ACCESS
		; address is not zero
		BRA		int_addressnotzero
		; address is zero
		dmx_setstate dmx_state_waitfordata
		BRA		int_addresszero
int_addressnotzero:
		MOVFF	addressh,countdownaddressh
		MOVFF	addressl,countdownaddressl
int_addresszero:
		LFSR	0,dmx_values_first
		BRA		int_end
int_invalidstartcode:
		; start code is not zero, so ignore all data up to the next received break
		dmx_setstate dmx_state_waitforbreak
		BRA		int_end

int_waitforaddress:
		DECF	countdownaddressl,F,ACCESS
		MOVLW	0
		SUBWFB	countdownaddressh,F,ACCESS
		; check if adress matches
		TSTFSZ	countdownaddressh,ACCESS
		; address doesn't match
		BRA		int_ignorebyte
		TSTFSZ	countdownaddressl,ACCESS
		; address doesn't match
		BRA		int_ignorebyte
		; address matches
		dmx_setstate dmx_state_waitfordata

int_waitfordata:
		MOVFF	RCREG,POSTINC0
		; check if this was the last value to be stored
		MOVLW	dmx_values_last
		; is the FSR index > dmx_values_last?
		CPFSGT	FSR0L,ACCESS
		BRA		int_end
		; this was the last value to be stored, so wait for a break again
		dmx_setstate dmx_state_waitforbreak

int_end:
		; return from interrupt and restore status registers
		MOVF	temp_wreg,W,ACCESS
		MOVFF	BSR,temp_bsr
		MOVFF	STATUS,temp_status
		RETFIE

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

init:	; do not use analog inputs
		MOVLW	0x0F
		MOVWF	ADCON1,ACCESS
		; clear watchdog timer.
		CLRWDT

init_vars:
		; initialize DMX values table
		CLRF	dmx_values_first + 0,ACCESS
		CLRF	dmx_values_first + 1,ACCESS
		CLRF	dmx_values_first + 2,ACCESS
		CLRF	dmx_values_first + 3,ACCESS
		CLRF	dmx_values_first + 4,ACCESS
		CLRF	dmx_values_first + 5,ACCESS
		CLRF	dmx_values_first + 6,ACCESS
		CLRF	dmx_values_first + 7,ACCESS
		CLRF	dmx_values_first + 8,ACCESS
		CLRF	dmx_values_first + 9,ACCESS
		CLRF	dmx_values_first + 10,ACCESS
		CLRF	dmx_values_first + 11,ACCESS
		; initialize DMX state
		dmx_setstate dmx_state_waitforsignal
		CLRF	dmx_signallostcountdown,ACCESS
		CLRF	switches1to4,ACCESS
		CLRF	switches5to12,ACCESS
		CLRF	addressh,ACCESS
		CLRF	addressl,ACCESS
		CLRF	outputoverride1to4,ACCESS
		CLRF	outputoverride5to12,ACCESS

init_switchmatrix:
		; disable all matrix send pins.
		MOVLW	0xFF
		MOVWF	LATB,ACCESS
		; set switch matrix send pins direction to output.
		MOVLW	B'11000000'
		MOVWF	TRISB,ACCESS
		; set switch matrix return pins direction to input.
		BSF		PORTE,RDPU,ACCESS
		MOVLW	B'01111111'
		MOVWF	TRISD,ACCESS

init_outputswitches:
		; initialize port C switches
		MOVLW	B'11111000'
		ANDWF	TRISC,F,ACCESS
		MOVLW	B'00000111'
		IORWF	LATC,F,ACCESS
		; initialize port E switches
		MOVLW	B'11111000'
		ANDWF	TRISE,F,ACCESS
		MOVLW	B'00000111'
		IORWF	LATE,F,ACCESS
		; initialize port A switches
		MOVLW	B'11000000'
		ANDWF	TRISA,F,ACCESS
		MOVLW	B'00111111'
		IORWF	LATA,F,ACCESS

init_uart:
		; set baud rate (250 kbps)
		CLRF	SPBRGH,ACCESS
		; if BRGH = 1 ==>
		; n = ((Fosc / baudrate) / 16) - 1 =
		; ((20000000 / 250000) / 4) - 1 = 20
		; MOVLW	19
		; if BRGH = 0 ==>
		; n = ((Fosc / baudrate) / 16) - 1 =
		; ((20000000 / 250000) / 16) - 1 = 4
		MOVLW	4
		MOVWF	SPBRG,ACCESS
		BSF		TRISC,7,ACCESS ; configure serial input pin
		BSF		TRISC,6,ACCESS ; configure serial output pin
		MOVLW	(1 << TX9) | (0 << TXEN) | (0 << SYNC) | (0 << SENDB) | (0 << BRGH) | (1 << TX9D)
		MOVWF	TXSTA,ACCESS
		MOVLW	(1 << SPEN) | (1 << RX9) | (0 << CREN) | (0 << ADDEN)
		MOVWF	RCSTA,ACCESS
		MOVLW	(0 << ABDOVF) | (1 << BRG16) | (0 << WUE) | (0 << ABDEN)
		MOVWF	BAUDCON,ACCESS
		; enable serial receive interrupt
		BSF		PIE1,RCIE,ACCESS
		; enable serial reception
		BSF		RCSTA,CREN,ACCESS

init_led:
		; enable power led.
		BCF		LATD,7,ACCESS
		; set power led pin direction to output.
		BCF		TRISD,7,ACCESS

init_timer0:
		; initialize timer 0: 16 bit timer with 1:32 prescale value for an interval of ~2,4 Hz
		MOVLW	(0 << TMR0ON) | (0 << T08BIT) | (0 << T0CS) | (0 << T0SE) | (0 << PSA) | (1 << T0PS2) | (0 << T0PS1) | (0 << T0PS0)
		MOVWF	T0CON,ACCESS
		; enable timer 0 interrupt
		BSF		INTCON,TMR0IE,ACCESS
		; enable timer 0
		BSF		T0CON,TMR0ON,ACCESS

init_interrupts:
		; clear watchdog timer.
		CLRWDT
		; enable perhipheral interrupts
		BSF		INTCON,PEIE,ACCESS
		; enable global interrupts
		BSF		INTCON,GIE,ACCESS

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

main:
do_input:
		CLRF	tempaddressl,ACCESS

		; activate "DMX address" switch matrix send 1
		MOVLW	~0x01
		MOVWF	LATB,ACCESS
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		; retrieve bits 2, 1 & 0 from PORTD
		MOVF	PORTD,W,ACCESS
		ANDLW	0x07
		; merge the three new bits with tempaddress
		MOVWF	tempaddressl,ACCESS

		; activate "DMX address" switch matrix send 2
		MOVLW	~0x02
		MOVWF	LATB,ACCESS
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		; retrieve bits 2, 1 & 0 from PORTD
		MOVF	PORTD,W,ACCESS
		ANDLW	0x07
		; merge the three new bits with tempaddress
		RLNCF	WREG,W,ACCESS
		RLNCF	WREG,W,ACCESS
		RLNCF	WREG,W,ACCESS
		IORWF	tempaddressl,F,ACCESS

		; activate "DMX address" switch matrix send 3
		MOVLW	~0x04
		MOVWF	LATB,ACCESS
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		; retrieve bits 2, 1 & 0 from PORTD
		MOVF	PORTD,W,ACCESS
		ANDLW	0x07
		; merge the three new bits with tempaddress
		RLNCF	WREG,W,ACCESS
		RLNCF	WREG,W,ACCESS
		RLNCF	WREG,W,ACCESS
		RLNCF	WREG,W,ACCESS
		RLNCF	WREG,W,ACCESS
		BCF		STATUS,C,ACCESS
		RLCF	WREG,W,ACCESS
		IORWF	tempaddressl,F,ACCESS

		COMF	tempaddressl,F,ACCESS

; temporarily disable global interrupts
		BCF		INTCON,GIE,ACCESS

		MOVLW	0
		BC		addressh_end
addressh_set:
		MOVLW	1
addressh_end:
		MOVWF	addressh,ACCESS

		MOVFF	tempaddressl,addressl

; reenable global interrupts
		BSF		INTCON,GIE,ACCESS

		CLRF	outputoverride1to4,ACCESS
		CLRF	outputoverride5to12,ACCESS

		; activate "output override" switch matrix send 1
		MOVLW	~0x08
		MOVWF	LATB,ACCESS
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		; retrieve bits 6, 5, 4 & 3 from PORTD
		MOVF	PORTD,W,ACCESS
		ANDLW	0x78
		RRNCF	WREG,W,ACCESS
		RRNCF	WREG,W,ACCESS
		RRNCF	WREG,W,ACCESS
		; merge the four new bits with outputoverride
		MOVWF	outputoverride5to12,ACCESS

		; activate "output override" switch matrix send 2
		MOVLW	~0x10
		MOVWF	LATB,ACCESS
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		; retrieve bits 6, 5, 4 & 3 from PORTD
		MOVF	PORTD,W,ACCESS
		ANDLW	0x78
		RRNCF	WREG,W,ACCESS
		RRNCF	WREG,W,ACCESS
		RRNCF	WREG,W,ACCESS
		; merge the four new bits with outputoverride
		RLNCF	WREG,W,ACCESS
		RLNCF	WREG,W,ACCESS
		RLNCF	WREG,W,ACCESS
		RLNCF	WREG,W,ACCESS
		IORWF	outputoverride5to12,F,ACCESS

		SETF	LATB,ACCESS

		; activate "output override" switch matrix send 3
		MOVLW	~0x20
		MOVWF	LATB,ACCESS
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		NOP
		; retrieve bits 6, 5, 4 & 3 from PORTD
		MOVF	PORTD,W,ACCESS
		ANDLW	0x78
		RRNCF	WREG,W,ACCESS
		RRNCF	WREG,W,ACCESS
		RRNCF	WREG,W,ACCESS
		; merge the four new bits with outputoverride
		MOVWF	outputoverride1to4,ACCESS

		SETF	LATB,ACCESS

		COMF	outputoverride1to4,F,ACCESS
		COMF	outputoverride5to12,F,ACCESS

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

do_processing:
		; clear watchdog timer.
		CLRWDT
		MOVLW	dmx_state_waitforsignal
		CPFSEQ	dmx_state,ACCESS
		BRA		do_processing_signalpresent
do_processing_nosignalpresent:
		; disable all switches.
		CLRF	switches1to4,ACCESS
		CLRF	switches5to12,ACCESS
		BRA		do_processing_end
do_processing_signalpresent:
		; set switches according to DMX buffer.
		MOVLW	127
		dmx_applyvalue 0,switches1to4,3
		dmx_applyvalue 1,switches1to4,2
		dmx_applyvalue 2,switches1to4,1
		dmx_applyvalue 3,switches1to4,0
		dmx_applyvalue 4,switches5to12,7
		dmx_applyvalue 5,switches5to12,6
		dmx_applyvalue 6,switches5to12,5
		dmx_applyvalue 7,switches5to12,4
		dmx_applyvalue 8,switches5to12,3
		dmx_applyvalue 9,switches5to12,2
		dmx_applyvalue 10,switches5to12,1
		dmx_applyvalue 11,switches5to12,0
do_processing_end:

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

do_output:
		; merge switched outputs with "output override" switch matrix
		MOVF	switches1to4,W,ACCESS
		IORWF	outputoverride1to4,W,ACCESS
		MOVWF	switches1to4temp,ACCESS

		MOVF	switches5to12,W,ACCESS
		IORWF	outputoverride5to12,W,ACCESS
		MOVWF	switches5to12temp,ACCESS

		; update switched output RC2
		BTFSC	switches1to4temp,3,ACCESS
		BRA		output1_set
		BSF		LATC,2,ACCESS
		BRA		output1_end
output1_set:
		BCF		LATC,2,ACCESS
output1_end:

		; update switched output RC1
		BTFSC	switches1to4temp,2,ACCESS
		BRA		output2_set
		BSF		LATC,1,ACCESS
		BRA		output2_end
output2_set:
		BCF		LATC,1,ACCESS
output2_end:

		; update switched output RC0
		BTFSC	switches1to4temp,1,ACCESS
		BRA		output3_set
		BSF		LATC,0,ACCESS
		BRA		output3_end
output3_set:
		BCF		LATC,0,ACCESS
output3_end:

		; update switched output RE2
		BTFSC	switches1to4temp,0,ACCESS
		BRA		output4_set
		BSF		LATE,2,ACCESS
		BRA		output4_end
output4_set:
		BCF		LATE,2,ACCESS
output4_end:

		; update switched output RE1
		BTFSC	switches5to12temp,7,ACCESS
		BRA		output5_set
		BSF		LATE,1,ACCESS
		BRA		output5_end
output5_set:
		BCF		LATE,1,ACCESS
output5_end:

		; update switched output RE0
		BTFSC	switches5to12temp,6,ACCESS
		BRA		output6_set
		BSF		LATE,0,ACCESS
		BRA		output6_end
output6_set:
		BCF		LATE,0,ACCESS
output6_end:

		; update switched output RA5
		BTFSC	switches5to12temp,5,ACCESS
		BRA		output7_set
		BSF		LATA,5,ACCESS
		BRA		output7_end
output7_set:
		BCF		LATA,5,ACCESS
output7_end:

		; update switched output RA4
		BTFSC	switches5to12temp,4,ACCESS
		BRA		output8_set
		BSF		LATA,4,ACCESS
		BRA		output8_end
output8_set:
		BCF		LATA,4,ACCESS
output8_end:

		; update switched output RA3
		BTFSC	switches5to12temp,3,ACCESS
		BRA		output9_set
		BSF		LATA,3,ACCESS
		BRA		output9_end
output9_set:
		BCF		LATA,3,ACCESS
output9_end:

		; update switched output RA2
		BTFSC	switches5to12temp,2,ACCESS
		BRA		output10_set
		BSF		LATA,2,ACCESS
		BRA		output10_end
output10_set:
		BCF		LATA,2,ACCESS
output10_end:

		; update switched output RA1
		BTFSC	switches5to12temp,1,ACCESS
		BRA		output11_set
		BSF		LATA,1,ACCESS
		BRA		output11_end
output11_set:
		BCF		LATA,1,ACCESS
output11_end:

		; update switched output RA0
		BTFSC	switches5to12temp,0,ACCESS
		BRA		output12_set
		BSF		LATA,0,ACCESS
		BRA		output12_end
output12_set:
		BCF		LATA,0,ACCESS
output12_end:

		BRA		main

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

delay:	MOVLW		0x06
		MOVWF		var1,ACCESS
delay1:	CLRF		var2,ACCESS
delay2:	CLRF		var3,ACCESS
delay3:	DECF		var3,F,ACCESS
		BTFSS		STATUS,Z
		BRA			delay3
		DECF		var2,F,ACCESS
		BTFSS		STATUS,Z
		BRA			delay2
		DECF		var1,F,ACCESS
		BTFSS		STATUS,Z
		BRA			delay1
		RETURN

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

chase1:
		CLRF	switches5to12,ACCESS

		MOVLW	0x08
		MOVWF	switches1to4,ACCESS
		CALL	do_output
		CALL	delay

		MOVLW	0x04
		MOVWF	switches1to4,ACCESS
		CALL	do_output
		CALL	delay

		MOVLW	0x02
		MOVWF	switches1to4,ACCESS
		CALL	do_output
		CALL	delay

		MOVLW	0x01
		MOVWF	switches1to4,ACCESS
		CALL	do_output
		CALL	delay

		CLRF	switches1to4,ACCESS

		MOVLW	0x80
		MOVWF	switches5to12,ACCESS
		CALL	do_output
		CALL	delay

		MOVLW	0x40
		MOVWF	switches5to12,ACCESS
		CALL	do_output
		CALL	delay

		MOVLW	0x20
		MOVWF	switches5to12,ACCESS
		CALL	do_output
		CALL	delay

		MOVLW	0x10
		MOVWF	switches5to12,ACCESS
		CALL	do_output
		CALL	delay

		MOVLW	0x08
		MOVWF	switches5to12,ACCESS
		CALL	do_output
		CALL	delay

		MOVLW	0x04
		MOVWF	switches5to12,ACCESS
		CALL	do_output
		CALL	delay

		MOVLW	0x02
		MOVWF	switches5to12,ACCESS
		CALL	do_output
		CALL	delay

		MOVLW	0x01
		MOVWF	switches5to12,ACCESS
		CALL	do_output
		CALL	delay

		RETURN

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

chase2:
		MOVLW	0xAA
		MOVWF	switches1to4,ACCESS
		MOVWF	switches5to12,ACCESS
		CALL	do_output
		CALL	delay

		MOVLW	0x55
		MOVWF	switches1to4,ACCESS
		MOVWF	switches5to12,ACCESS
		CALL	do_output
		CALL	delay

		RETURN

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

		end
